/**
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.airavata.registry.core.app.catalog.resources;

import org.apache.airavata.common.exception.ApplicationSettingsException;
import org.apache.airavata.registry.core.app.catalog.model.AppModuleMapping;
import org.apache.airavata.registry.core.app.catalog.model.AppModuleMapping_PK;
import org.apache.airavata.registry.core.app.catalog.model.ApplicationInterface;
import org.apache.airavata.registry.core.app.catalog.model.ApplicationModule;
import org.apache.airavata.registry.core.app.catalog.util.AppCatalogJPAUtils;
import org.apache.airavata.registry.core.app.catalog.util.AppCatalogQueryGenerator;
import org.apache.airavata.registry.core.app.catalog.util.AppCatalogResourceType;
import org.apache.airavata.registry.cpi.AppCatalogException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.persistence.EntityManager;
import javax.persistence.Query;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class AppModuleMappingAppCatalogResourceAppCat extends AppCatAbstractResource {
    private final static Logger logger = LoggerFactory.getLogger(AppModuleMappingAppCatalogResourceAppCat.class);
    private String interfaceId;
    private String moduleId;
    private AppInterfaceResource appInterfaceResource;
    private AppModuleResource moduleResource;


    public String getModuleId() {
        return moduleId;
    }

    public void setModuleId(String moduleId) {
        this.moduleId = moduleId;
    }

    public String getInterfaceId() {
        return interfaceId;
    }

    public void setInterfaceId(String interfaceId) {
        this.interfaceId = interfaceId;
    }

    public AppInterfaceResource getAppInterfaceResource() {
        return appInterfaceResource;
    }

    public void setAppInterfaceResource(AppInterfaceResource appInterfaceResource) {
        this.appInterfaceResource = appInterfaceResource;
    }

    public AppModuleResource getModuleResource() {
        return moduleResource;
    }

    public void setModuleResource(AppModuleResource moduleResource) {
        this.moduleResource = moduleResource;
    }

    @Override
    public void remove(Object identifier) throws AppCatalogException {
        HashMap<String, String> ids;
        if (identifier instanceof Map){
            ids = (HashMap)identifier;
        }else {
            logger.error("Identifier should be a map with the field name and it's value");
            throw new AppCatalogException("Identifier should be a map with the field name and it's value");
        }
        EntityManager em = null;
        try {
            em = AppCatalogJPAUtils.getEntityManager();
            em.getTransaction().begin();
            AppCatalogQueryGenerator generator= new AppCatalogQueryGenerator(APP_MODULE_MAPPING);
            generator.setParameter(AppModuleMappingConstants.INTERFACE_ID, ids.get(AppModuleMappingConstants.INTERFACE_ID));
            if (ids.get(AppModuleMappingConstants.MODULE_ID) != null){
                generator.setParameter(AppModuleMappingConstants.MODULE_ID, ids.get(AppModuleMappingConstants.MODULE_ID));
            }
            Query q = generator.deleteQuery(em);
            q.executeUpdate();
            em.getTransaction().commit();
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
        } catch (ApplicationSettingsException e) {
            logger.error(e.getMessage(), e);
            throw new AppCatalogException(e);
        } finally {
            if (em != null && em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
        }
    }

    public void removeAll() throws AppCatalogException {
        EntityManager em = null;
        try {
            em = AppCatalogJPAUtils.getEntityManager();
            em.getTransaction().begin();
            AppCatalogQueryGenerator generator= new AppCatalogQueryGenerator(APP_MODULE_MAPPING);
            Query q = generator.deleteQuery(em);
            q.executeUpdate();
            em.getTransaction().commit();
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
        } catch (ApplicationSettingsException e) {
            logger.error(e.getMessage(), e);
            throw new AppCatalogException(e);
        } finally {
            if (em != null && em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
        }
    }

    @Override
    public AppCatalogResource get(Object identifier) throws AppCatalogException {
        HashMap<String, String> ids;
        if (identifier instanceof Map){
            ids = (HashMap)identifier;
        }else {
            logger.error("Identifier should be a map with the field name and it's value");
            throw new AppCatalogException("Identifier should be a map with the field name and it's value");
        }
        EntityManager em = null;
        try {
            em = AppCatalogJPAUtils.getEntityManager();
            em.getTransaction().begin();
            AppCatalogQueryGenerator generator = new AppCatalogQueryGenerator(APP_MODULE_MAPPING);
            generator.setParameter(AppModuleMappingConstants.INTERFACE_ID, ids.get(AppModuleMappingConstants.INTERFACE_ID));
            generator.setParameter(AppModuleMappingConstants.MODULE_ID, ids.get(AppModuleMappingConstants.MODULE_ID));
            Query q = generator.selectQuery(em);
            AppModuleMapping result = (AppModuleMapping) q.getSingleResult();
            AppModuleMappingAppCatalogResourceAppCat resource =
                    (AppModuleMappingAppCatalogResourceAppCat) AppCatalogJPAUtils.getResource(AppCatalogResourceType.APP_MODULE_MAPPING, result);
            em.getTransaction().commit();
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
            return resource;
        } catch (ApplicationSettingsException e) {
            logger.error(e.getMessage(), e);
            throw new AppCatalogException(e);
        } finally {
            if (em != null && em.isOpen()) {
                if (em.getTransaction().isActive()) {
                    em.getTransaction().rollback();
                }
                em.close();
            }
        }
    }

    @Override
    public List<AppCatalogResource> get(String fieldName, Object value) throws AppCatalogException {
        List<AppCatalogResource> resourceList = new ArrayList<AppCatalogResource>();
        EntityManager em = null;
        try {
            em = AppCatalogJPAUtils.getEntityManager();
            em.getTransaction().begin();
            Query q;
            AppCatalogQueryGenerator generator = new AppCatalogQueryGenerator(APP_MODULE_MAPPING);
            List results;
            if (fieldName.equals(AppModuleMappingConstants.INTERFACE_ID)) {
                generator.setParameter(AppModuleMappingConstants.INTERFACE_ID, value);
                q = generator.selectQuery(em);
                results = q.getResultList();
                if (results.size() != 0) {
                    for (Object result : results) {
                        AppModuleMapping moduleMapping = (AppModuleMapping) result;
                        AppModuleMappingAppCatalogResourceAppCat resource =
                                (AppModuleMappingAppCatalogResourceAppCat) AppCatalogJPAUtils.getResource(AppCatalogResourceType.APP_MODULE_MAPPING, moduleMapping);
                        resourceList.add(resource);
                    }
                }
            } else if (fieldName.equals(AppModuleMappingConstants.MODULE_ID)) {
                generator.setParameter(AppModuleMappingConstants.MODULE_ID, value);
                q = generator.selectQuery(em);
                results = q.getResultList();
                if (results.size() != 0) {
                    for (Object result : results) {
                        AppModuleMapping moduleMapping = (AppModuleMapping) result;
                        AppModuleMappingAppCatalogResourceAppCat resource =
                                (AppModuleMappingAppCatalogResourceAppCat) AppCatalogJPAUtils.getResource(AppCatalogResourceType.APP_MODULE_MAPPING, moduleMapping);
                        resourceList.add(resource);
                    }
                }
            }else {
                em.getTransaction().commit();
                if (em.isOpen()) {
                    if (em.getTransaction().isActive()){
                        em.getTransaction().rollback();
                    }
                    em.close();
                }
                logger.error("Unsupported field name for app module mapping resource.", new IllegalArgumentException());
                throw new IllegalArgumentException("Unsupported field name for app module mapping resource.");
            }
            em.getTransaction().commit();
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new AppCatalogException(e);
        } finally {
            if (em != null && em.isOpen()) {
                if (em.getTransaction().isActive()) {
                    em.getTransaction().rollback();
                }
                em.close();
            }
        }
        return resourceList;
    }

    @Override
    public List<AppCatalogResource> getAll() throws AppCatalogException {
        return null;
    }

    @Override
    public List<String> getAllIds() throws AppCatalogException {
        return null;
    }

    @Override
    public List<String> getIds(String fieldName, Object value) throws AppCatalogException {
        logger.error("Unsupported for objects with a composite identifier");
        throw new AppCatalogException("Unsupported for objects with a composite identifier");
    }

    @Override
    public void save() throws AppCatalogException {
        EntityManager em = null;
        try {
            em = AppCatalogJPAUtils.getEntityManager();
            AppModuleMapping existngModuleMap = em.find(AppModuleMapping.class, new AppModuleMapping_PK(interfaceId, moduleId));
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }

            em = AppCatalogJPAUtils.getEntityManager();
            em.getTransaction().begin();
            ApplicationInterface applicationInterface = em.find(ApplicationInterface.class, interfaceId);
            ApplicationModule applicationModule = em.find(ApplicationModule.class, moduleId);
            if (existngModuleMap !=  null){
                existngModuleMap.setApplicationInterface(applicationInterface);
                existngModuleMap.setApplicationModule(applicationModule);
                em.merge(existngModuleMap);
            }else {
                AppModuleMapping appModuleMapping = new AppModuleMapping();
                appModuleMapping.setInterfaceID(interfaceId);
                appModuleMapping.setApplicationInterface(applicationInterface);
                appModuleMapping.setModuleID(moduleId);
                appModuleMapping.setApplicationModule(applicationModule);
                em.persist(appModuleMapping);
            }
            em.getTransaction().commit();
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new AppCatalogException(e);
        } finally {
            if (em != null && em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
        }
    }

    @Override
    public boolean isExists(Object identifier) throws AppCatalogException {
        HashMap<String, String> ids;
        if (identifier instanceof Map){
            ids = (HashMap)identifier;
        }else {
            logger.error("Identifier should be a map with the field name and it's value");
            throw new AppCatalogException("Identifier should be a map with the field name and it's value");
        }
        EntityManager em = null;
        try {
            em = AppCatalogJPAUtils.getEntityManager();
            AppModuleMapping moduleMapping = em.find(AppModuleMapping.class,
                    new AppModuleMapping_PK(ids.get(AppModuleMappingConstants.INTERFACE_ID),
                            ids.get(AppModuleMappingConstants.MODULE_ID)));
            if (em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
            return moduleMapping != null;
        } catch (ApplicationSettingsException e) {
            logger.error(e.getMessage(), e);
            throw new AppCatalogException(e);
        } finally {
            if (em != null && em.isOpen()) {
                if (em.getTransaction().isActive()){
                    em.getTransaction().rollback();
                }
                em.close();
            }
        }
    }
}